$(document).ready(function () {
    var cart = {};
    var productDetails = {}; // Per memorizzare i dettagli completi del prodotto

    // Non carichiamo più i metodi di pagamento dinamicamente
    // perché ci sono problemi di serializzazione
    
    // Evento per i pulsanti di categoria
    $('.category-button').on('click', function() {
        var categoryId = $(this).data('category');
        loadProductsByCategory(categoryId);
    });

    function loadProductsByCategory(categoryId) {
        $.ajax({
            url: '/GetProductsByCategory',
            type: 'GET',
            data: { categoryId: categoryId },
            success: function(data) {
                updateProductsGrid(data);
            },
            error: function(error) {
                console.error('Errore nel caricamento dei prodotti:', error);
            }
        });
    }

    function updateProductsGrid(products) {
        var productsGrid = $('.products-grid');
        productsGrid.empty();

        products.forEach(function(product) {
            var productImagePath = product.Logos ? serverUrl + '/Content/Files/Product/' + product.Logos : '/path/to/default/image.jpg';
            
            var productCard = $(
                '<div class="product-card" data-product-id="' + product.ProductCode + '" data-category-id="' + product.Category_ID + '">' +
                '<img src="' + productImagePath + '" alt="' + product.ShortDescription + '" class="card-img-top" />' +
                '<div class="card-body">' +
                '<h5 class="card-title product-name">' + product.ShortDescription + '</h5>' +
                '<p class="card-text product-price">' + product.SellPrice.toFixed(2) + '</p>' +
                '</div>' +
                '</div>'
            );
            
            productsGrid.append(productCard);
            
            // Memorizza i dettagli completi del prodotto
            productDetails[product.ProductCode] = {
                id: product.ProductCode,
                name: product.ShortDescription,
                price: product.SellPrice,
                vatRate: product.VatRate
            };
        });
    }

    // Rimosso il metodo loadPaymentModes per usare valori statici
    // nel template HTML invece

    function addToCart(productId, productName, productPrice) {
        console.log('Aggiungo al carrello:', productId, productName, productPrice);
        
        // Verifica che i dati siano validi
        if (!productId || !productName || isNaN(productPrice)) {
            console.error('Dati prodotto non validi per carrello:', {id: productId, name: productName, price: productPrice});
            return;
        }
        
        if (cart[productId]) {
            cart[productId].quantity += 1;
            // Verifica che la quantità sia un numero
            if (isNaN(cart[productId].quantity)) {
                cart[productId].quantity = 1;
            }
        } else {
            // Assicurati che il prezzo sia un numero
            if (isNaN(productPrice)) productPrice = 0;
            
            cart[productId] = {
                id: productId,
                name: productName,
                price: productPrice,
                quantity: 1,
                vatRate: productDetails[productId] && productDetails[productId].vatRate ? 
                         productDetails[productId].vatRate : 22 // Default a IVA 22%
            };
        }

        console.log('Carrello aggiornato:', cart);
        renderCart();
    }

    function removeFromCart(productId) {
        delete cart[productId];
        renderCart();
    }

    function renderCart() {
        var total = 0;
        var cartTableBody = $('#products-table tbody');
        cartTableBody.empty();

        for (var id in cart) {
            var product = cart[id];
            var row = '<tr>' +
                '<td>' + product.quantity + 'x</td>' +
                '<td>' + product.name + '</td>' +
                '<td>' + (product.price * product.quantity).toFixed(2) + '</td>' +
                '<td class="remove-column"><i class="fas fa-trash remove-icon" data-product-id="' + id + '"></i></td>' +
                '</tr>';
            cartTableBody.append(row);
            total += parseFloat(product.price) * product.quantity;
        }

        $('#cart-total').text(total.toFixed(2));
    }

    $(document).on('click', '.product-card', function () {
        var productId = $(this).data('product-id') || '';
        var productName = $(this).find('.product-name').text() || 'Prodotto';
        var productPrice = parseFloat($(this).find('.product-price').text()) || 0;
        
        // Verifica che i dati siano validi
        if (productId && productPrice > 0) {
            addToCart(productId, productName, productPrice);
        } else {
            console.error('Dati prodotto non validi:', {id: productId, name: productName, price: productPrice});
        }
    });

    $(document).on('click', '.remove-icon', function () {
        var productId = $(this).data('product-id');
        removeFromCart(productId);
    });

    $('#checkoutBtn').click(function () {
        var total = $('#cart-total').text();
        $('#checkout-total').text(total);
        $('#checkoutModal').modal('show');
    });

    $('#confirmCheckout').click(function () {
        var paymentType = $('#paymentType').val() || "1";
        var paymentTypeText = $('#paymentType option:selected').text() || "Contanti";
        var totalAmount = parseFloat($('#cart-total').text()) || 0;
        
        // Assicuriamoci che il valore sia numerico
        if (isNaN(totalAmount)) totalAmount = 0;
        
        // Verifica se ci sono prodotti nel carrello
        if (Object.keys(cart).length === 0) {
            alert('Carrello vuoto! Aggiungi prodotti prima di procedere.');
            return;
        }
        
        // Verifica che il metodo di pagamento sia valido
        paymentType = paymentType || "1"; // Default a contanti se non selezionato
        
        console.log('Contenuto del carrello:', cart);
        console.log('Metodo di pagamento:', paymentType, paymentTypeText);
        console.log('Totale:', totalAmount);
        
        // Prepara i dati dello scontrino come array e assicuriamoci che
        // tutti i campi richiesti siano presenti con nomi di proprietà corretti
        var cartItems = [];
        for (var id in cart) {
            // Crea una copia dei dati del carrello per evitare problemi
            // IMPORTANTE: Rispettare ESATTAMENTE i nomi delle proprietà nel C# (Id, Name, etc.)
            var item = {
                Id: cart[id].id || id,
                Name: cart[id].name || "Prodotto",
                Price: parseFloat(cart[id].price) || 0,
                Quantity: parseInt(cart[id].quantity) || 1,
                VatRate: parseFloat(cart[id].vatRate) || 22
            };
            cartItems.push(item);
        }
        
        // Verifica che ci siano effettivamente elementi
        if (cartItems.length === 0) {
            alert('Errore: nessun prodotto valido nel carrello');
            return;
        }
        
        // Formato dati definitivo che deve rispettare esattamente la struttura attesa dal server
        var receiptData = {
            Items: cartItems,           // IMPORTANTE: Rispetta la proprietà maiuscola "Items" del DTO C#
            TotalAmount: totalAmount || 0,  // IMPORTANTE: Rispetta la proprietà maiuscola "TotalAmount"
            PaymentMethod: paymentType   // IMPORTANTE: Rispetta la proprietà maiuscola "PaymentMethod"
        };
        
        console.log('Dati scontrino:', JSON.stringify(receiptData));
        
        // Mostra indicatore di caricamento
        $('#confirmCheckout').prop('disabled', true).text('Emissione in corso...');
        
        // Invia al server
        var jsonData = JSON.stringify(receiptData);
        console.log("JSON ESATTO da inviare:", jsonData);
        console.log("Items count:", receiptData.Items.length);
        
        // Per debug - visualizziamo il payload esatto
        console.log("DUMP COMPLETO DEL PAYLOAD:");
        for (var i = 0; i < receiptData.Items.length; i++) {
            console.log("Item " + i + ":", 
                "ID:", receiptData.Items[i].Id,
                "Name:", receiptData.Items[i].Name,
                "Price:", receiptData.Items[i].Price,
                "Quantity:", receiptData.Items[i].Quantity,
                "VatRate:", receiptData.Items[i].VatRate
            );
        }
        
        $.ajax({
            url: '/PrintReceipt',
            type: 'POST',
            contentType: 'application/json; charset=utf-8',
            dataType: 'json',
            data: jsonData,
            success: function(response) {
                console.log('Risposta server:', response);
                if (response && response.success) {
                    alert('Scontrino emesso con successo!');
                    // Svuota il carrello
                    cart = {};
                    renderCart();
                    $('#checkoutModal').modal('hide');
                } else {
                    var message = response && response.message ? response.message : 'Errore sconosciuto';
                    alert('Errore: ' + message);
                }
            },
            error: function(xhr, status, error) {
                console.error('Errore AJAX:', xhr, status, error);
                console.error('Dettagli risposta:', xhr.responseText);
                
                let errorMsg = 'Errore durante l\'emissione dello scontrino';
                
                try {
                    // Prova a interpretare la risposta come JSON
                    var jsonResponse = JSON.parse(xhr.responseText);
                    if (jsonResponse && jsonResponse.message) {
                        errorMsg += ': ' + jsonResponse.message;
                    } else if (error) {
                        errorMsg += ': ' + error;
                    }
                } catch (e) {
                    // Se non è JSON, mostra il messaggio di errore grezzo
                    if (xhr.responseText) {
                        errorMsg += ': ' + xhr.responseText;
                    } else if (error) {
                        errorMsg += ': ' + error;
                    }
                }
                
                alert(errorMsg);
            },
            complete: function() {
                $('#confirmCheckout').prop('disabled', false).text('Conferma');
            }
        });
    });

    // Funzione per la ricerca prodotti per codice a barre
    $('#barcodeSearch').on('keypress', function(e) {
        if (e.which === 13) { // Tasto Enter
            var barcode = $(this).val().trim();
            if (barcode) {
                searchProductByBarcode(barcode);
                $(this).val(''); // Pulisci il campo
            }
        }
    });
    
    function searchProductByBarcode(barcode) {
        $.ajax({
            url: '/GetProductsByBarcode',
            type: 'GET',
            data: { barcode: barcode },
            success: function(data) {
                if (data && data.length > 0) {
                    // Aggiungi automaticamente al carrello
                    var product = data[0];
                    addToCart(
                        product.ProductCode, 
                        product.ShortDescription, 
                        product.SellPrice
                    );
                    
                    // Memorizza dettagli
                    productDetails[product.ProductCode] = {
                        id: product.ProductCode,
                        name: product.ShortDescription,
                        price: product.SellPrice,
                        vatRate: product.VatRate
                    };
                } else {
                    alert('Prodotto non trovato!');
                }
            },
            error: function(error) {
                console.error('Errore nella ricerca del prodotto:', error);
            }
        });
    }
});