﻿// cart.js - Funzioni per la gestione del carrello

let isBarcodeScanning = false;
let barcodeBuffer = '';
let barcodeTimeout = null;

// Configura il timeout per la scansione (in millisecondi)
const BARCODE_SCAN_TIMEOUT = 300; // 300ms di inattività dopo un carattere per considerare la fine del codice

// Ascolta gli eventi keydown per iniziare la scansione
document.addEventListener('keydown', function (event) {
    console.log("keydown ", event.key);
    const activeElement = document.activeElement;
    const isInputField = activeElement.tagName === 'INPUT' && (activeElement.type === 'text' || activeElement.type === 'number');
    var key = '';
    if (event.key === 'ArrowDown') {
        event.preventDefault();
    }

    // Inizia la scansione se viene premuto '*'
    if (event.key === '*' && !isBarcodeScanning) {
        isBarcodeScanning = true;  // Inizia la scansione
        barcodeBuffer = '';  // Resetta il buffer
        event.preventDefault();  // Evita che l'asterisco venga inserito nei campi
    } else if (isBarcodeScanning && !isInputField) {
        if (event.key === 'Shift') {
             key = '';
        } else {
             key = event.key;
        }
        
        // Se viene premuto "Enter", il barcode è completato
        if (key === 'Enter') {
            completeBarcodeScan();
        } else {
            barcodeBuffer += key;  // Aggiungi il carattere al buffer
            // Resetta il timeout se un nuovo carattere è stato inserito
            clearTimeout(barcodeTimeout);
            barcodeTimeout = setTimeout(completeBarcodeScan, BARCODE_SCAN_TIMEOUT);  // Timeout per completare la scansione
        }
        event.preventDefault();  // Evita l'inserimento nei campi se è in corso la scansione
    }
});

// Ascolta gli eventi keypress per costruire il barcode
document.addEventListener('keypress', function (event) {
    console.log("keypress ", event.key);
    if (event.key === 'Shift') {
        event.preventDefault();
    }
    if (event.key === 'ArrowDown') {
        event.preventDefault();
    }

    if (isBarcodeScanning) {
        const key = event.key;

        // Se viene premuto "Enter", il barcode è completato
        if (key === 'Enter') {
            completeBarcodeScan();
        } else {
            barcodeBuffer += key;  // Aggiungi il carattere al buffer

            // Resetta il timeout se un nuovo carattere è stato inserito
            clearTimeout(barcodeTimeout);
            barcodeTimeout = setTimeout(completeBarcodeScan, BARCODE_SCAN_TIMEOUT);  // Timeout per completare la scansione
        }

        event.preventDefault();  // Blocca l'inserimento nei campi attivi
    }
});

// Funzione per completare la scansione del barcode
function completeBarcodeScan() {
    //if (barcodeBuffer.startsWith('*')) {
        const productCode = barcodeBuffer.trim();  // Rimuovi gli spazi indesiderati
        if (productCode.length === 0) {
            console.error('Codice a barre vuoto.');
            resetBarcodeScan();
            return;
        }
        console.log("Prodotto letto dal barcode:", productCode);

        // Inserisci qui la logica per cercare e gestire il prodotto in base al codice letto
        processBarcode(productCode);
    //} else {
    //    console.error('Il codice a barre deve iniziare con un asterisco (*)');
    //}

    resetBarcodeScan();
}

function resetBarcodeScan() {
    // Resetta lo stato della scansione
    isBarcodeScanning = false;
    barcodeBuffer = '';
    clearTimeout(barcodeTimeout);  // Cancella il timeout se la scansione è stata completata
}
function addProductToCart(prodid, descr, prices, vatrep) {
    console.log("Aggiunta prodotto al carrello:", { prodid, descr, prices, vatrep });

    const cart = document.getElementById('cart');
    if (!cart) {
        console.error("Elemento carrello non trovato nel DOM.");
        return;
    }

    const priceValue = parseFloat(prices);
    if (isNaN(priceValue) || priceValue <= 0) {
        console.error("Prezzo non valido rilevato:", priceValue);
        return;
    }
    console.log("Prezzo originale:", prices, "Prezzo convertito:", priceValue);

    const existingProduct = cart.querySelector(`div[data-id="${prodid}"]`);
    if (existingProduct) {
        const qtyField = existingProduct.querySelector('.quantity');
        qtyField.value = parseInt(qtyField.value) + 1;
        updateTotalPrice(existingProduct);
        document.querySelector(`input[name="raws-Qty[]"][data-id="${prodid}"]`).value = qtyField.value;
    } else {
        const cartItem = document.createElement('div');
        cartItem.className = 'cart-item';
        cartItem.setAttribute('data-id', prodid);
        cartItem.setAttribute('data-vatrep', vatrep);

        cartItem.innerHTML = `
            <div class="row align-items-center">
                <div class="col-12">
                    <span class="descr">${descr}</span>
                </div>
                <div class="col-4">
                    <div class="form-group">
                        <label>Prezzo:</label>
                        <input type="number" class="price form-control" value="${priceValue.toFixed(2)}" step="0.01" min="0" onchange="updateTotalPrice(this.closest('.cart-item'))">
                    </div>
                </div>
                <div class="col-4">
                    <div class="form-group">
                        <label>Q.tà:</label>
                        <input type="number" class="quantity form-control" value="1" min="1" onchange="updateTotalPrice(this.closest('.cart-item'))">
                    </div>
                </div>
                <div class="col-4">
                    <div class="form-group">
                        <label>Sconto (%):</label>
                        <input type="number" class="discount form-control" value="0" min="0" max="100" onchange="updateTotalPrice(this.closest('.cart-item'))">
                    </div>
                </div>
                <div class="col-4">
                    <span class="total">€ ${priceValue.toFixed(2)}</span>
                </div>
                <div class="col-4">
                    &nbsp;
                </div>
                <div class="col-4 text-center">
                    <button class="remove-btn" onclick="removeProductFromCart('${prodid}')">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
            <!-- Aggiungi il campo IVA nascosto -->
            <input type="hidden" class="raws-vat" name="raws-vat[]" value="${vatrep}" data-id="${prodid}">
        `;

        cart.appendChild(cartItem);

        const cart1 = document.getElementById('cart');
        console.log("ScrollTop:", cart1.scrollTop, "ScrollHeight:", cart1.scrollHeight, "ClientHeight:", cart1.clientHeight);

        const lastCartItem = cart1.lastElementChild; // Seleziona l'ultimo elemento

        if (lastCartItem) {
            lastCartItem.scrollIntoView({ behavior: 'smooth' }); // Scrolla fino all'ultimo elemento
        }
        const hiddenInputsContainer = document.getElementById('cart-hidden-inputs');
        if (hiddenInputsContainer) {
            const hiddenInputs = `
                <input type="hidden" name="raws[]" value="${prodid}" data-id="${prodid}">
                <input type="hidden" name="descr[]" value="${descr}" data-id="${prodid}">
                <input type="hidden" name="raws-Qty[]" value="1" data-id="${prodid}">
                <input type="hidden" name="raws-price[]" value="${priceValue.toFixed(2)}" data-id="${prodid}">
                <input type="hidden" name="raws-vat[]" value="${vatrep}" data-id="${prodid}">
                <input type="hidden" name="raws-discount[]" value="0" data-id="${prodid}">
            `;
            hiddenInputsContainer.insertAdjacentHTML('beforeend', hiddenInputs);
        } else {
            console.error("Elemento 'cart-hidden-inputs' non trovato nel DOM.");
        }
    }

    updateOverallTotal();
    saveCartToLocalStorage();
}

function addProductToCartFromLocalStorage(item) {
    const cart = document.getElementById('cart');
    if (!cart) {
        console.error("Elemento carrello non trovato nel DOM.");
        return;
    }

    const existingProduct = cart.querySelector(`div[data-id="${item.prodid}"]`);
    if (existingProduct) {
        existingProduct.querySelector('.quantity').value = item.qty;
        existingProduct.querySelector('.price').value = item.price.toFixed(2);
        existingProduct.querySelector('.discount').value = item.discount || 0;
        updateTotalPrice(existingProduct);
    } else {
        const cartItem = document.createElement('div');
        cartItem.className = 'cart-item';
        cartItem.setAttribute('data-id', item.prodid);
        cartItem.setAttribute('data-vatrep', item.vatrep);

        const discountedPrice = item.price * (1 - (item.discount || 0) / 100);
        const totalPrice = discountedPrice * item.qty;

        cartItem.innerHTML = `
            <div class="row align-items-center">
                <div class="col-12">
                    <span class="descr">${item.descr}</span>
                </div>
                <div class="col-4">
                    <div class="form-group">
                        <label>Prezzo:</label>
                        <input type="number" class="price form-control" value="${item.price.toFixed(2)}" step="0.01" min="0" onchange="updateTotalPrice(this.closest('.cart-item'))">
                    </div>
                </div>
                <div class="col-4">
                    <div class="form-group">
                        <label>Q.tà:</label>
                        <input type="number" class="quantity form-control" value="${item.qty}" min="1" onchange="updateTotalPrice(this.closest('.cart-item'))">
                    </div>
                </div>
                <div class="col-4">
                    <div class="form-group">
                        <label>Sconto (%):</label>
                        <input type="number" class="discount form-control" value="${item.discount || 0}" min="0" max="100" onchange="updateTotalPrice(this.closest('.cart-item'))">
                    </div>
                </div>
                <div class="col-4 text-center">
                    <span class="total">€ ${totalPrice.toFixed(2)}</span>
                </div>
                <div class="col-4">
                    &nbsp;
                </div>
                <div class="col-4 text-center">
                    <button class="remove-btn" onclick="removeProductFromCart('${item.prodid}')">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
            </div>
            <!-- Aggiungi il campo IVA nascosto -->
            <input type="hidden" class="raws-vat" name="raws-vat[]" value="${item.vatrep}" data-id="${item.prodid}">
        `;

        const cart = document.getElementById('cart');
        console.log("ScrollTop:", cart.scrollTop, "ScrollHeight:", cart.scrollHeight, "ClientHeight:", cart.clientHeight);

        const lastCartItem = cart.lastElementChild; // Seleziona l'ultimo elemento

        if (lastCartItem) {
            lastCartItem.scrollIntoView({ behavior: 'smooth' }); // Scrolla fino all'ultimo elemento
        }
    }
}



function updateTotalPrice(cartItem) {
    const priceInput = cartItem.querySelector('.price');
    const quantityInput = cartItem.querySelector('.quantity');
    const discountInput = cartItem.querySelector('.discount');

    const price = parseFloat(priceInput.value);
    const qty = parseInt(quantityInput.value);
    const discount = parseFloat(discountInput.value) || 0;

    // Verifica dei valori validi
    if (isNaN(price) || price < 0) {
        alert('Inserisci un prezzo valido.');
        priceInput.value = '0.00';
        return;
    }
    if (isNaN(qty) || qty < 1) {
        alert('La quantità deve essere almeno 1.');
        quantityInput.value = '1';
        return;
    }
    if (isNaN(discount) || discount < 0 || discount > 100) {
        alert('Inserisci uno sconto valido (0-100).');
        discountInput.value = '0';
        return;
    }

    // Calcola il prezzo totale dell'elemento con lo sconto
    const discountedPrice = price * (1 - discount / 100);
    const totalPrice = discountedPrice * qty;

    // Aggiorna il prezzo totale dell'elemento
    cartItem.querySelector('.total').textContent = `€ ${totalPrice.toFixed(2)}`;

    // Aggiorna i campi nascosti
    const prodid = cartItem.getAttribute('data-id');
    document.querySelector(`input[name="raws-Qty[]"][data-id="${prodid}"]`).value = qty;
    document.querySelector(`input[name="raws-price[]"][data-id="${prodid}"]`).value = price.toFixed(2);
    document.querySelector(`input[name="raws-discount[]"][data-id="${prodid}"]`).value = discount.toFixed(2);

    // Aggiorna il totale generale
    updateOverallTotal();
    saveCartToLocalStorage();
}


function updateOverallTotal() {
    let totalAmount = 0;
    let totalQuantity = 0; // Variabile per il totale della quantità

    document.querySelectorAll('.cart-item').forEach(function (item) {
        const price = parseFloat(item.querySelector('.price').value);
        const qty = parseInt(item.querySelector('.quantity').value);
        const discount = parseFloat(item.querySelector('.discount').value) || 0;

        const discountedPrice = price * (1 - discount / 100);
        const itemTotal = discountedPrice * qty;

        if (!isNaN(itemTotal)) {
            totalAmount += itemTotal;
            totalQuantity += qty; // Aggiunge la quantità
        } else {
            console.error('Invalid item total detected:', itemTotal);
        }
    });

    // Aggiorna il testo del totale e la quantità
    const totalAmountElement = document.getElementById('total-amount');
    if (totalAmountElement) {
        totalAmountElement.textContent = totalAmount.toFixed(2).replace('.', ','); 
    }

    const totalButton = document.querySelector('.btn-total'); // Seleziona il pulsante
    if (totalButton) {
        totalButton.innerHTML = `TOT. <span id="total-amount">€ ${totalAmount.toFixed(2).replace('.', ',')}</span> (${totalQuantity} pz)`; // Aggiorna il testo con la quantità di pezzi
    } else {
        console.error("Elemento 'btn-total' non trovato nel DOM.");
    }
    console.log("Calcolo totale ", totalAmount);
    // Aggiorna i campi nascosti del totale
    $('#TotalSalePrice').val(totalAmount.toFixed(2));
    $('#TotalPriceNoDiscount').val(totalAmount.toFixed(2));
}



// Funzione per pulire il carrello dopo la vendita
function clearCart() {
    document.querySelectorAll('.cart-item').forEach(item => item.remove());
    updateOverallTotal();
    saveCartToLocalStorage(); // Pulisce il carrello nel localStorage

    console.log("Carrello svuotato.");
}

let productToDeleteId = null; // Variabile per tenere traccia del prodotto da eliminare

// Funzione per aprire la modale di conferma eliminazione
function removeProductFromCart(prodid) {
    productToDeleteId = prodid; // Memorizza l'ID del prodotto da eliminare
    $('#confirmDeletionModal').modal('show'); // Mostra la modale
}

// Funzione per confermare l'eliminazione del prodotto
function confirmDeleteProduct() {
    const cart = document.getElementById('cart');
    const productToRemove = cart.querySelector(`div[data-id="${productToDeleteId}"]`);
    if (productToRemove) {
        // Rimuove l'elemento visivamente dal carrello
        productToRemove.remove();
        // Rimuove anche i campi nascosti associati al prodotto
        document.querySelectorAll(`input[data-id="${productToDeleteId}"]`).forEach(input => input.remove());

        // Aggiorna il carrello nel Local Storage
        saveCartToLocalStorage();
    }
    updateOverallTotal();
    $('#confirmDeletionModal').modal('hide'); // Nascondi la modale
}

// Aggiungi un event listener al pulsante di conferma della modale
document.getElementById('confirmDeleteBtn').addEventListener('click', confirmDeleteProduct);

function confirmSale(totalAmount, paymentType) {
    const receivedAmount = parseFloat($('#receivedAmount').val()) || 0;

    // Verifica se il pagamento è in contanti e l'importo ricevuto è sufficiente
    if (paymentType === 'Contanti' && parseFloat(receivedAmount) < totalAmount) {
        $('#receivedAmount').addClass('is-invalid'); // Evidenzia il campo se insufficiente
        $('#changeAmount').text(`Resto da dare: € 0.00`);
        showMessage("error", "Attenzione", "L'importo ricevuto è insufficiente!"); // Messaggio di errore
        $('#receivedAmount').val(0); // Imposta il campo a zero per un nuovo inserimento
        return; // Blocca l'operazione se il pagamento è insufficiente
    }

    // Se il pagamento è sufficiente o è elettronico, rimuovi l'evidenziazione
    $('#receivedAmount').removeClass('is-invalid');
    $('#confirmSaleModal').modal('hide');

    // Completa la vendita chiamando la funzione `completeSale`
    completeSale(totalAmount, receivedAmount, paymentType);
}

// Funzione per calcolare il resto
function calculateChange() {
    const totalAmount = parseFloat($('#total-amount').text().replace('€', '').trim());
    const receivedAmount = parseFloat($('#receivedAmount').val());

    // Controlla se l'importo ricevuto è minore dell'importo totale
    if (isNaN(receivedAmount) || receivedAmount ==0) {
        $('#changeAmount').text(`Resto da dare: € 0.00`); // Imposta a zero se l'importo ricevuto non è valido
        return;
    }

    if (receivedAmount < totalAmount) {
        $('#changeAmount').text(`Resto da dare: € 0.00`); // Non ci sarà resto se non si ha l'importo totale
        $('#receivedAmount').addClass('is-invalid'); // Aggiungi classe per evidenziare il campo
        // Usa la libreria per mostrare il messaggio di errore
        showMessage("error", "Attenzione", "L\'importo ricevuto è insufficiente!"); // Messaggio di errore
        $('#receivedAmount').val(0);
        return;
    } else {
        $('#receivedAmount').removeClass('is-invalid'); // Rimuovi l'evidenziazione se il valore è valido
        const change = receivedAmount - totalAmount;
        $('#changeAmount').text(`Resto da dare: € ${change.toFixed(2)}`);
    }
}


// Funzione per mostrare il modal di conferma
//function showConfirmSaleModal(totalAmount, paymentTypeText) {
//    let receivedAmountField = '';

//    // Mostra il campo di importo ricevuto solo se il pagamento è in contanti
//    if (paymentTypeText === 'Contanti') {
//        receivedAmountField = `
//            <div class="form-group">
//                <label>Importo ricevuto:</label>
//                <input type="number" id="receivedAmount" class="form-control"
//                       value="${totalAmount.toFixed(2)}" step="0.01" min="0"
//                       onchange="calculateChange()"
//                       onclick="this.select();"> <!-- Seleziona il contenuto al click -->
//            </div>
//            <p>Resto da dare: <strong id="changeAmount">€ ${totalAmount.toFixed(2)}</strong></p>
//        `;
//    }

//    $('#confirmSaleModal .modal-body').html(`
//        <p>Conferma la vendita:</p>
//        <p>Importo totale: <strong>€ ${totalAmount.toFixed(2)}</strong></p>
//        <p>Tipo di pagamento: <strong>${paymentTypeText}</strong></p>
//        ${receivedAmountField}
//    `);

//    $('#confirmSaleModal').css('z-index', 1050).modal('show');
//}

function showConfirmSaleModal(totalAmount, paymentType) {
    let receivedAmountField = '';

    // Mostra il campo di importo ricevuto solo se il pagamento è in contanti
    if (paymentType === 'Contanti') {
        receivedAmountField = `
            <div class="form-group">
                <label>Importo ricevuto:</label>
                <input type="number" id="receivedAmount" class="form-control" 
                       value="${totalAmount.toFixed(2)}" step="0.01" min="0" 
                       onchange="calculateChange()" 
                       onclick="this.select();"> 
            </div>
            <p>Resto da dare: <strong id="changeAmount">€ ${totalAmount.toFixed(2)}</strong></p>
        `;
    }

    $('#confirmSaleModal .modal-body').html(`
        <p>Conferma la vendita:</p>
        <p>Importo totale: <strong>€ ${totalAmount.toFixed(2)}</strong></p>
        <p>Tipo di pagamento: <strong>${paymentType}</strong></p>
        ${receivedAmountField}
    `);

    // Cambia il comportamento del pulsante in base al tipo di pagamento
    $('#confirmSaleButton').off('click').on('click', function () {
        confirmSale(totalAmount, paymentType);
    });

    $('#confirmSaleModal').css('z-index', 1050).modal('show');
}



// Funzione per la gestione della vendita

function SaleEnd(paymode) {
    const overlayElement = document.getElementById('overlay');
    if (overlayElement) {
        overlayElement.style.display = 'flex';
        overlayElement.style.zIndex = '1040';
    } else {
        console.error('Elemento "overlay" non trovato nel DOM.');
        return;
    }

    // Estrai il testo e rimuovi simboli non numerici (tranne la virgola e il punto)
    let totalAmountText = $('#total-amount').text().trim();
    //console.log("Valore di #total-amount:", totalAmountText);

    // Rimuove tutti i simboli tranne i numeri, il punto e la virgola
    totalAmountText = totalAmountText.replace(/[^\d,.-]/g, '');
    //console.log("Totale text pulito: ", totalAmountText);

    // Sostituisci la virgola con un punto per garantire la corretta conversione
    const totalAmount = parseFloat(totalAmountText.replace(',', '.'));

    //console.log("Totale convertito: ", totalAmount);

    if (isNaN(totalAmount) || totalAmount <= 0) {
        showMessage('warning', 'Errore', 'Il valore della vendita deve essere maggiore di zero!');
        overlayElement.style.display = 'none';
        return;
    }

    //console.log("Totale Vendita", totalAmount);
    $('#modalTotalAmount').text(totalAmount.toFixed(2));

    let paymentTypeText = paymode === 0 ? 'Contanti' : paymode === 1 ? 'Elettronico' : 'Sconosciuto';

    showConfirmSaleModal(totalAmount, paymentTypeText);

   

   //$('#confirmSaleButton').off('click').on('click', function () {
        
   // });
 
    $('#confirmSaleModal').on('hidden.bs.modal', function () {
        overlayElement.style.display = 'none';
        $('#pleaseWaitDiv').modal('hide');
    });
}

function completeSale(totalAmount, receivedAmount, paymode) {
    // Mostra l'overlay di caricamento
    const loadingOverlay = document.createElement('div');
    loadingOverlay.id = 'loadingOverlay';
    loadingOverlay.style.position = 'fixed';
    loadingOverlay.style.top = '0';
    loadingOverlay.style.left = '0';
    loadingOverlay.style.width = '100%';
    loadingOverlay.style.height = '100%';
    loadingOverlay.style.backgroundColor = 'rgba(0, 0, 0, 0.5)';
    loadingOverlay.style.color = '#fff';
    loadingOverlay.style.display = 'flex';
    loadingOverlay.style.justifyContent = 'center';
    loadingOverlay.style.alignItems = 'center';
    loadingOverlay.style.zIndex = '2000';
    loadingOverlay.innerHTML = '<p>Attendere, elaborazione in corso...</p>';
    document.body.appendChild(loadingOverlay);

    // Nasconde la modale ma mantiene l'overlay
    $('#confirmSaleModal').modal('hide');

    // Esegui la vendita con il tipo di pagamento selezionato
    if (navigator.onLine) {
        processOnlineSale(paymode)
            .then(() => {
                // Rimuove l'overlay al completamento della vendita
                removeLoadingOverlay();
            })
            .catch(() => {
                // Gestione dell'errore con rimozione dell'overlay
                removeLoadingOverlay();
            });
    } else {
        processOfflineSale(paymode)
            .then(() => {
                // Rimuove l'overlay al completamento della vendita
                removeLoadingOverlay();
            })
            .catch(() => {
                // Gestione dell'errore con rimozione dell'overlay
                removeLoadingOverlay();
            });
    }
    // Logica per completare la vendita
    console.log("Vendita completata con successo:", totalAmount, receivedAmount);
    // Esegui la logica di invio dati al server o gestione del pagamento
}
// Funzione per rimuovere l'overlay di caricamento
function removeLoadingOverlay() {
    const loadingOverlay = document.getElementById('loadingOverlay');
    if (loadingOverlay) {
        document.body.removeChild(loadingOverlay);
    }
}
